﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.Tilemaps;

public class BasicMovement : MonoBehaviour
{
    public Animator animator;
    public float SpeedMultiplier = 1.0f;

    private Rigidbody2D _Body;
    private TileAudio _TileAudio;
    private Tilemap _BaseTilemapBack;
    private Tilemap _BaseTilemapMiddle;
    
    private float _XInput, _YInput;

    public bool CanMove { get; set; } = true;

    /// <summary>
    /// Plays the correct movement animation based on the player's movement vector
    /// </summary>
    /// <param name="moveVector">A vector of the player's movement</param>
    private void PlayMovementAnimation(Vector2 moveVector)
    {
        if(moveVector.x == 0.0f && moveVector.y == 0.0f)
        {
            animator.Play("Idle", -1, 0.0f);
        }

        else if(moveVector.x < -0.01f)
        {
            animator.Play("WalkLeft");
        }

        else if(moveVector.x >= 0.01f)
        {
            animator.Play("WalkRight");
        }

        else if (moveVector.y < -0.01f)
        {
            animator.Play("WalkFront");
        }

        else if (moveVector.y >= 0.01f)
        {
            animator.Play("WalkBack");
        }
    }

    void Start()
    {
        //Get the components
        _Body = GetComponent<Rigidbody2D>();
        _BaseTilemapBack = GameObject.Find("baseTilemapBack").GetComponent<Tilemap>();
        _BaseTilemapMiddle = GameObject.Find("baseTilemapMiddle").GetComponent<Tilemap>();
        _TileAudio = _BaseTilemapBack.gameObject.GetComponent<TileAudio>();
    }

    void LateUpdate()
    {
        if (CanMove)
        {
            if (Input.GetAxisRaw("Horizontal") != 0.0f || Input.GetAxisRaw("Vertical") != 0.0f)
            {
                //We're using analog inputs, get the input and set the values into variables
                _XInput = Input.GetAxisRaw("Horizontal");
                _YInput = Input.GetAxisRaw("Vertical");
            }

            else
            {
                //We're using the d-pad, get the input and set the values into variables
                _XInput = Input.GetAxisRaw("Controller D-Pad X");
                _YInput = Input.GetAxisRaw("Controller D-Pad Y");
            }

            if (Mathf.Abs(_XInput) > Mathf.Abs(_YInput))
            {
                //We only allow direct inputs -  if X is greater, use that normalized
                _XInput = Utilities.NormalizeInput(_XInput);
                _YInput = 0.0f;
            }

            else
            {
                //We only allow direct inputs -  if Y is greater, use that normalized
                _XInput = 0.0f;
                _YInput = Utilities.NormalizeInput(_YInput);
            }

            if (_XInput != 0 || _YInput != 0)
            {
                //We're moving, calculate the new position, move the body to there and play the correct animation
                Vector2 moveVector = new Vector2(_XInput, _YInput);
                Vector2 newPos = new Vector2((transform.position.x + moveVector.x * SpeedMultiplier * Time.fixedDeltaTime), transform.position.y + moveVector.y * SpeedMultiplier * Time.fixedDeltaTime);
                _Body.MovePosition(newPos);
                PlayMovementAnimation(moveVector);

                //Get the tile the player is on the base back
                TileBase tile = _BaseTilemapBack.GetTile(_BaseTilemapBack.WorldToCell(newPos));

                if (tile != null)
                {
                    //We've got a tile on the base back, try play the audio
                    _TileAudio.PlayTileAudio(tile.name);
                }

                else
                {
                    //Get the tile the player is on the base middle
                    tile = _BaseTilemapMiddle.GetTile(_BaseTilemapMiddle.WorldToCell(newPos));

                    if (tile != null)
                    {
                        //We've got a tile on the base middle, try play the audio
                        _TileAudio.PlayTileAudio(tile.name);
                    }
                }
            }

            else
            {
                //We've stopped moving - play the animation and fade out movement
                PlayMovementAnimation(new Vector2(0.0f, 0.0f));
                _TileAudio.TryFadeOutTileAudio();
            }
        }
    }
}
